document.addEventListener('alpine:init', () => {
    Alpine.data('alert', () => ({
        show: false,
        alert: '',
        close: false,

        closeAlert() {
            if (this.show) {
                this.show = false;
                this.$refs.alert.classList.add('tjmodal-out')
                setTimeout(() => {
                    this.$refs.alert.classList.remove('tjmodal-out')
                    this.$refs.alert.style.zIndex = '0'
                    this.$refs.alert.style.position = 'absolute'
                    this.$refs.alert.style.left = '-9999999px'
                }, 400);
            }
        },

        showAlert($event) {
            this.$refs.alert.style.zIndex = '11000'
            this.$refs.alert.style.position = ''
            this.$refs.alert.style.left = ''
            setTimeout(() => {
                this.show = true;
                this.alert = $event.detail[0];
            }, 10);
            if ($event.detail[0].type != 'info') {
                setTimeout(() => {
                    this.closeAlert()
                }, 1500);
            }
        }
    }))

    Alpine.data('account', (show_balance) => ({
        show_balance: true,

        init() {
            this.show_balance = show_balance
        },

        async toggleVisibility() {
            const hiddenElements = document.querySelectorAll('.hidden-balance')
            const shownElements = document.querySelectorAll('.show-balance')
            hiddenElements.forEach(el => el.classList.toggle('d-none'))
            shownElements.forEach(el => el.classList.toggle('d-none'))

            const { show_balance } = await (await fetch(route('account.preferences', { show_balance: !this.show_balance }))).json()
            this.show_balance = show_balance
        }
    }))

    Alpine.data('send_coin', (price) => ({
        fiat_amount: '0.00',
        getFiat() {
            if (!isNaN(this.$wire.form.amount)) {
                let fiat_amount = price * this.$wire.form.amount
                this.fiat_amount = number_format(fiat_amount, 4)
            }
        },

        async paste() {
            try {
                const text = await navigator.clipboard.readText();
                this.$wire.form.receiver = text

                return text;
            } catch (err) {
                console.error("Failed to paste: ", err);
                return null;
            }
        },

        scan() {
            this.$dispatch('alert', [{
                type: 'info',
                text: `The QR code crypto address barcode scanner cannot pair with your device at the moment. Please type or copy and paste your crypto address.`,
            }])
        },

        shorten(address) {
            if (address?.length <= 12) return address;
            return `${address?.substring(0, 12)}...${address?.substring(address.length - 12)}`;
        },

        max(balance) {
            this.$wire.form.amount = balance
            this.getFiat()
        },

        percentify(percent, balance) {
            this.$wire.form.amount = ((percent / 100) * balance).toFixed(4)
            this.getFiat()
        },

        closeSheet() {
            document.querySelector('.bottomSheet').classList.remove("bottomSheet--offScreen")
            document.querySelector('.bottomSheet').classList.add("bottomSheet--onScreen")
        }
    }))

    Alpine.data('receive', () => ({
        copy(text) {
            navigator.clipboard.writeText(text).then(() => {
                this.$dispatch('alert', [{
                    type: 'notice',
                    text: 'Copied',
                }])
            }).catch(error => {
                this.$dispatch('alert', [{
                    type: 'notice',
                    text: `Failed to copy text: ${error}`,
                }])
            });
        }
    }))

    Alpine.directive('qr', (el, { expression }, { evaluate }) => {
        new QRCode(el, evaluate(expression));
    })

    Alpine.data('buy', () => ({
        amount: 150,
        coin_amount: '0.0000',
        payment_method: 'apple_pay',
        provider: 'Transak',
        price: 0,
        minimum_buy: 0,

        set(price, minimum_buy) {
            this.price = price
            this.minimum_buy = minimum_buy
            this.updateFiat()
        },

        updateFiat() {
            this.coin_amount = number_format(this.amount / this.price, 4)
        },

        changeProvider(provider) {
            this.provider = provider;
            closeBottomSheet('#buy');
        },

        title_case(str) {
            return str
                .replace(/_/g, ' ')
                .toLowerCase()
                .replace(/\b\w/g, char => char.toUpperCase())
        },

        buy() {
            if (this.amount < this.minimum_buy) {
                return this.$dispatch('alert', [{
                    type: 'info',
                    text: `The minimum amount you can buy is $${this.minimum_buy}`,
                }])
            }
            switch (this.provider) {
                case 'binance':
                    location.href = 'https://www.binance.com/crypto/buy'
                    break;

                case 'transak':
                    location.href = 'https://global.transak.com/'
                    break;

                case 'moon_pay':
                    location.href = 'https://global.transak.com/'
                    break;

                default:
                    break;
            }
        }
    }))

    Alpine.data('swap', () => ({
        from: {
            price: '',
            fiat_amount: '0.00'
        },

        to: {
            price: '',
            coin_amount: '0',
            fiat_amount: '0.00',
        },
        validated: false,

        updateDetails() {
            if (this.$el.value.length >= 7) {
                this.$el.value = this.$el.value.slice(0, this.$el.value.length - 1)
                return false
            }
            let from_fiat_amount = this.from.price * this.$wire.amount
            this.from.fiat_amount = number_format(from_fiat_amount, 2)

            let coin_amount = from_fiat_amount / this.to.price

            this.to.coin_amount = coin_amount.toFixed(4)
            this.to.fiat_amount = number_format(this.to.coin_amount * this.to.price, 2)
        },

        percentify(percent, balance) {
            this.$wire.amount = ((percent / 100) * balance).toFixed(4)
            this.updateDetails()
        },
    }))

    Alpine.data('manage', () => ({
        active: null,
        change() {
            this.$dispatch('loading')
            this.active ? this.$el.classList.remove('active') : this.$el.classList.add('active')
            this.$wire.change()
        }
    }))

    Alpine.data('mode', () => ({
        dark: null,
        change() {
            const color = this.dark ? 'light' : 'dark';
            document.querySelector('html').setAttribute('data-bs-theme', color)

            this.dark ? this.$el.classList.remove('active') : this.$el.classList.add('active')
            this.dark = !this.dark
            this.$wire.change()
        }
    }))

    Alpine.store('actions', {
        show_fee_notice(siteName = 'Trustjex', fee = 'network') {
            dispatchAlert(`The ${fee} charges a transaction fee which varies based on blockchain usage. ${siteName} does not charge any fee. 0% of these fees are paid to ${siteName}.`)
        },

        async copy(string) {
            try {
                await navigator.clipboard.writeText(string);
                dispatchAlert(`Copied.`, 'notice')
            } catch (err) {
                dispatchAlert(`Failed to copy. ${err}`, 'error')
            }
        }
    })

    Alpine.data('earn', () => ({
        earning: '0.00',
        increment: 0,

        select(plan_id, increment) {
            this.increment = increment
            this.$wire.select(plan_id)
        },

        update() {            
            if(!isNaN(this.$wire.amount)) {
                this.earning = number_format(this.$wire.amount * this.increment, 2)
            }
        }
    }))

    Alpine.data('verification', () => ({
        validate(event) {
            const fileName = event.target.files[0].name;
            const allowed = ['png', 'jpg', 'jpeg', 'pdf'];

            let fileExtension = fileName.split('.').pop().toLowerCase();
            let extensionAllowed = allowed.some(extension => extension === fileExtension);

            if (!extensionAllowed) {
                return dispatchAlert('File type not allowed, only pdf files and images allowed.')
            }
        }
    }))
})