<?php

namespace App\Models;

use App\Observers\CoinObserver;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

#[ObservedBy(CoinObserver::class)]
class Coin extends Model
{
    use HasFactory;

    protected $guarded = ['id'];
    
    protected $casts = [
        'is_active' => 'boolean',
    ];

    public function details() {
        return $this->hasOne(CoinDetail::class);
    }

    public function balances() {
        return $this->hasMany(UserBalance::class);
    }

    public function settings() {
        return $this->hasMany(UserSetting::class);
    }

    public static function active() {
        $user = auth_user();
        return static::where('is_active', true)
                     ->whereHas('settings', function($settings) use($user) {
                         $settings->where('is_active', true)->where('user_id', $user->id);
                });
    }

    protected function userBalance(): Attribute {
        return Attribute::make(get: fn() => 
                                $this->balances()
                                ->where('user_id', auth()->id())
                                ->first()
                            );
    }

    protected function userSettings(): Attribute {
        return Attribute::make(get: fn() => 
                                $this->settings()
                                ->where('user_id', auth()->id())
                                ->first()
                            );
    }

    protected function user(): Attribute {
        
        return Attribute::make(
            get: fn() => (object) [
                'fiat_balance' => setting('currency').number_format($this->user_balance->value * $this->details->price, 2),
                'balance' => $this->user_balance->value,
                'settings' => $this->user_settings,
                'transactions' => $this->transactions()->where('user_id', auth()->id())->get(),
                'investments' => $this->investments()?->where('user_id', auth()->id())->get(),
            ],
        );
    }

    protected function hasSibling(): Attribute {
        return Attribute::make(get: function() {
            $siblingCoin = $this->where('symbol', $this->symbol)->get();
            if(count($siblingCoin) > 1) {
                return true;
            }
        });
    }

    public function network() {
        return $this->belongsTo(Network::class);
    }

    protected function networkCoin(): Attribute {
        return Attribute::make(get: fn() => $this->where('symbol', $this->network->symbol)->first());
    }

    protected function fullName(): Attribute {
        return Attribute::make(fn() => $this->has_sibling ? "$this->name {$this->network->name}" : $this->name);
    }

    public function transactions() {
        return $this->hasMany(Transaction::class);
    }

    public function shortAddress(): Attribute {
        return Attribute::make(get: fn() => shorten_address($this->wallet_address));
    }

    public function plan() {
        return $this->hasOne(Plan::class);
    }

    public function investments() {
        return $this->plan?->investments();
    }
}
