<?php

namespace App\Livewire\Coin;

use App\Models\Coin;
use App\Models\User;
use App\Utils\HandlesSwap;
use Livewire\Component;

class Swap extends Component
{
    use HandlesSwap;

    public Coin $from_coin;
    public Coin $to_coin;
    public Coin $network_coin;

    public $amount;
    public $network_fee;


    public function render() {
        return view('livewire.coin.swap', [
            'coins' => Coin::active()->get()->sortByDesc(function($coin) {
                return $this->amountInFiat($coin->user->balance, $coin);
            }),
        ])->layout('components.layouts.account-inner', ['title' => 'Swap']);
    }

    public function mount(Coin $coin) {
        $this->from_coin = $coin->name ? $coin : Coin::active()->first();
        $this->to_coin = Coin::active()->where('id', '!=', $this->from_coin->id)->inRandomOrder()->first();
    }

    public function change(int $coin_id, string $which = 'from_coin') {
        $coin = Coin::find($coin_id);
        $opposite = $which == 'from_coin' ? 'to_coin':  'from_coin';

        if($this->$which->id != $this->$opposite->id) {
            $this->$which = $coin;
            $details = $this->updateDetails();

            return $this->dispatch('changed-coin', $details);
        }

        return $this->dispatchAlert('Please, select another coin.', 'notice');
    }

    public function validateDetails() {
        if(($this->amount <= 0 )) {
            return $this->dispatchAlert('Please, enter an amount.', 'notice');
        }

        if($this->amountIsLessMinimumSwap()) {
            return $this->dispatchAlert($this->lessThanMinimumSwap());
        }

        if($this->balanceIsInsufficient()) {
            return $this->dispatchAlert($this->balanceInsufficient());
        }

        $this->network_fee = $this->fee(
            type: 'swap',
            coin: $this->from_coin,
            amount: $this->amount,
        );

        return $this->dispatch('swap-validated');
    }
    
    public function swap() {
        $this->initValidation($this->from_coin, $this->amount);

        if($this->cannotSwapCoin()) {
            return $this->dispatchAlert($this->coinSendNotAllowed('swap'));
        }

        if ($this->networkFeeIsInsufficient()) {
            return $this->dispatchAlert($this->insufficientFee(
                type:'swap', 
                fee: number_format($this->network_fee->coin, 4),
                coin: $this->from_coin,
            ));
        }

        $fromCoinBalance = $this->from_coin->user_balance;
        $toCoinBalance = $this->to_coin->user_balance;
        $networkCoinBalance = $this->from_coin->network_coin->user_balance;
        $toAmountInCoin = $this->coinAmountInCoin($this->amount, $this->from_coin, $this->to_coin);

        $fromCoinBalance->decrement('value', $this->amount);
        $networkCoinBalance->decrement('value', $this->network_fee->coin);
        $toCoinBalance->increment('value', $toAmountInCoin);
        
        $transaction = $this->createTransaction();
        
        $this->dispatchAlert('Swap completed.', 'success');

        return $this->redirect(route('account.transaction', $transaction->hash), true);
    }
}
