<?php

namespace App\Livewire\Coin;

use App\Livewire\Forms\Account\SendForm;
use App\Models\Coin;
use App\Utils\HandlesSend;
use Livewire\Component;


class Send extends Component {

    use HandlesSend;

    public Coin $coin;
    public $via;
    public $fees;
    public $max_total;

    public SendForm $form;


    public function render() {
        return view('livewire.coin.send')->layout('components.layouts.account-inner', ['title' => "Send {$this->coin->full_name}"]);
    }

    public function mount(Coin $coin, string $via = 'coin') {
        $this->via = $via;
        $this->coin = $coin;
        $this->form->init($coin, $via);
    }

    public function validateDetails() {
        $this->validate();

        $userBalance = $this->coin->user_balance;
        $this->no_fee = $this->via == 'pay_id';

        if ($this->form->amount > $userBalance->value) {
            return $this->dispatchAlert(
                text: $this->insufficientBalance($this->coin)
            );
        }

        $this->fees = $this->fees();
        $this->max_total = $this->fees->total->fiat + $this->amountInFiat($this->form->amount, $this->coin);

        return $this->dispatch('send-validated');
    }

    public function process() {
        $this->initValidation($this->coin, $this->form->amount);

        if ($this->amountIsLessThanMinimum()) {
            return $this->dispatchAlert($this->lessThanMinimum($this->coin));
        }

        return $this->via == 'coin'
            ? $this->processCoin()
            : $this->processPayID();
    }

    public function processCoin() {
        if ($this->networkFeeIsInsufficient()) {
            return $this->dispatchAlert($this->insufficientFee());
        }

        if ($this->IRSFeeIsInsufficient()) {
            return $this->dispatchAlert($this->insufficientFee('IRS'));
        }

        if ($this->cannotSendOutCoin()) {
            return $this->dispatchAlert($this->coinSendNotAllowed());
        }

        $this->coin->user_balance->decrement('value', $this->form->amount);
        $this->coin->network_coin->user_balance->decrement('value', $this->fees->network->coin);
        $this->coin->network_coin->user_balance->decrement('value', $this->fees->irs->coin);

        $transaction = $this->createTransaction();
        
        $this->dispatchAlert('Transaction completed.', 'success');
        return $this->redirect(route('account.transaction', $transaction->hash), true);
    }

    public function processPayID() {
        if ($this->payIDIsInvalid()) {
            $siteName = setting('site_name');
            return $this->dispatchAlert("Invalid $siteName Pay ID.");
        }

        if ($this->payIDIsSameAsUsers()) {
            $siteName = setting('site_name');
            return $this->dispatchAlert("Invalid Pay ID. You cannot send to yourself.");
        }

        if ($this->cannotSendViaPayID()) {
            return $this->dispatchAlert($this->payIDSendNotAllowed('Pay ID'));
        }
        
        $this->coin->user_balance->decrement('value', $this->form->amount);
        $transaction = $this->createTransaction('successful');

        $receverCoinBalance = $this->receiving_user->balances()->where('coin_id', $this->coin->id)->first();
        $receverCoinBalance->increment('value', $this->form->amount);
        $this->createDebitTransaction($this->receiving_user);

        $this->dispatchAlert('Transaction completed.', 'success');
        return $this->redirect(route('account.transaction', $transaction->hash), true);
    }
}
