<?php

namespace App\Filament\Resources;

use Filament\Forms;
use App\Models\Coin;
use App\Models\Plan;
use Filament\Tables;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Filament\Resources\Resource;
use Filament\Forms\Components\Section;
use Filament\Tables\Actions\BulkAction;
use Illuminate\Database\Eloquent\Model;
use Filament\Forms\Components\TextInput;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use App\Filament\Resources\PlanResource\Pages;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Resources\PlanResource\RelationManagers;

class PlanResource extends Resource {
    protected static ?string $model = Plan::class;

    protected static ?string $navigationIcon = 'heroicon-o-cube';
    public static ?string $navigationGroup = 'Plan Management';

    public static function form(Form $form): Form {
        return $form
            ->schema([
                Section::make('Coin, Minimum and Maximum')->schema([
                    Forms\Components\Select::make('coin_id')
                        ->relationship('coin')
                        ->preload()
                        ->searchable()
                        ->native(false)
                        ->required()
                        ->live()
                        ->getOptionLabelFromRecordUsing(fn(Model $coin) => $coin->full_name),

                    Forms\Components\TextInput::make('minimum_investment')
                        ->suffix(fn(Get $get) => Coin::find($get('coin_id'))?->symbol)
                        ->numeric()
                        ->required()
                        ->maxLength(255)
                        ->default(500),

                    Forms\Components\TextInput::make('maximum_investment')
                        ->suffix(fn(Get $get) => Coin::find($get('coin_id'))?->symbol)
                        ->numeric()
                        ->required()
                        ->maxLength(255)
                        ->default(0),

                    Forms\Components\Toggle::make('is_active')
                        ->required(),
                ])->columns(3),

                Section::make('Increments and Duration')->schema([
                    Forms\Components\TextInput::make('increment')
                        ->afterStateHydrated(fn(TextInput $component, $state) => $component->state($state * 100))
                        ->dehydrateStateUsing(fn(string $state) => $state / 100)
                        ->numeric()
                        ->required()
                        ->suffix('%')
                        ->hint('Between 0 - 100%')
                        ->minValue(1)
                        ->maxValue(100),

                    Forms\Components\Select::make('increment_interval')
                        ->options([
                            'secondly' => 'Every Second',
                            'minutely' => 'Every Minute',
                            'hourly' => 'Every Hour',
                            'daily' => 'Every Day',
                            'monthly' => 'Every Month',
                            'yearly' => 'Every Year',
                        ])
                        ->native(false)
                        ->searchable()
                        ->required(),

                    Forms\Components\TextInput::make('duration')
                        ->hint('Format: 1 Days, 2 Days, 1 Hours, 3 Hours etc. Care should be taken when chaning this setting, possible program error may occur. ')
                        ->required()
                        ->maxLength(255)
                        ->columnSpanFull(),
                ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('coin.full_name')
                    ->searchable(['name', 'symbol'])
                    ->sortable(['name', 'symbol']),

                Tables\Columns\TextColumn::make('full_minimum')
                    ->label('Minimum Investment')
                    ->sortable(['minimum_investment'])
                    ->searchable(['minimum_investment']),

                Tables\Columns\TextColumn::make('full_maximum')
                    ->sortable(['maximum_investment'])
                    ->label('Maximum Investment')
                    ->searchable(['maximum_investment']),

                Tables\Columns\IconColumn::make('is_active')
                    ->sortable()
                    ->boolean(),

                Tables\Columns\TextColumn::make('increment')
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('increment_interval')
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('duration')
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    BulkAction::make('Mark Plans as Active')
                        ->requiresConfirmation()
                        ->action(function (Collection $records) {
                            $records->each(function ($record) {
                                $record->is_active = true;
                                $record->save();
                            });
                        })
                        ->icon('heroicon-o-check-circle')
                        ->color('primary'),

                    BulkAction::make('Mark Plans as Inactive')
                        ->requiresConfirmation()
                        ->action(function (Collection $records) {
                            $records->each(function ($record) {
                                $record->is_active = false;
                                $record->save();
                            });
                        })
                        ->icon('heroicon-o-x-circle')
                        ->color('danger'),
                ]),
            ]);
    }

    public static function getRelations(): array {
        return [
            //
        ];
    }

    public static function getPages(): array {
        return [
            'index' => Pages\ListPlans::route('/'),
            'create' => Pages\CreatePlan::route('/create'),
            'view' => Pages\ViewPlan::route('/{record}'),
            'edit' => Pages\EditPlan::route('/{record}/edit'),
        ];
    }
}
