<?php

namespace App\Filament\Resources;

use App\Filament\Resources\InvestmentResource\Pages;
use App\Models\Investment;
use Filament\Forms;
use Filament\Forms\Components\Section;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\BulkAction;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

class InvestmentResource extends Resource {
    protected static ?string $model = Investment::class;

    protected static ?string $navigationIcon = 'heroicon-o-chart-bar';

    protected static ?string $navigationGroup = 'Plan Management';

    public static function form(Form $form): Form {
        return $form
            ->schema([
                Section::make('Investment Details')->schema([
                    Forms\Components\Select::make('user_id')
                        ->relationship('user', 'email')
                        ->native(false)
                        ->preload()
                        ->searchable()
                        ->required(),

                    Forms\Components\Select::make('plan_id')
                        ->relationship('plan')
                        ->native(false)
                        ->preload()
                        ->searchable()
                        ->required()
                        ->getOptionLabelFromRecordUsing(fn(Model $plan) => $plan->coin->full_name),

                    Forms\Components\TextInput::make('amount')
                        ->required()
                        ->numeric(),

                    Forms\Components\Toggle::make('is_active')
                        ->required(),
                ])->columns(3),

                Section::make('Expiration and Date Started')->schema([
                    Forms\Components\DateTimePicker::make('started_at')
                        ->required(),

                    Forms\Components\DateTimePicker::make('expires_at')
                        ->required(),
                ])->columns(),
            ]);
    }

    public static function table(Table $table): Table {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('user.email')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('plan.coin.full_name')
                    ->searchable(['name', 'symbol'])
                    ->sortable(['name', 'symbol']),
                Tables\Columns\TextColumn::make('full_amount')
                    ->label('Amount')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\IconColumn::make('is_active')
                    ->sortable()
                    ->boolean(),
                Tables\Columns\TextColumn::make('started_at')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('expires_at')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Action::make('deactivate')->action(function (Investment $record) {
                    $record->is_active = false;
                    $record->save();
                })->visible(fn(Investment $record): bool => $record->is_active),

                Action::make('activate')->action(function (Investment $record) {
                    $record->is_active = true;
                    $record->save();
                })->visible(fn(Investment $record): bool => !$record->is_active),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    BulkAction::make('Mark Investments as Active')
                        ->requiresConfirmation()
                        ->action(function (Collection $records) {
                            $records->each(function ($record) {
                                $record->is_active = true;
                                $record->save();
                            });
                        })
                        ->icon('heroicon-o-check-circle')
                        ->color('primary'),

                    BulkAction::make('Mark Investments as Inactive')
                        ->requiresConfirmation()
                        ->action(function (Collection $records) {
                            $records->each(function ($record) {
                                $record->is_active = false;
                                $record->save();
                            });
                        })
                        ->icon('heroicon-o-x-circle')
                        ->color('danger'),
                ]),
            ]);
    }

    public static function getRelations(): array {
        return [
            //
        ];
    }

    public static function getPages(): array {
        return [
            'index' => Pages\ListInvestments::route('/'),
            'create' => Pages\CreateInvestment::route('/create'),
            'view' => Pages\ViewInvestment::route('/{record}'),
            'edit' => Pages\EditInvestment::route('/{record}/edit'),
        ];
    }
}
